//------------------------------------------------------------------------------
// File: svr_slgameobjmods.cs
// This file is used for methods and data that modify/act on game objects
// Author: Matthew Rudge
//------------------------------------------------------------------------------

// Game object global modifier types. If adding more, then update the maximum
// count as well.
$OSMOD_HITPOINTS   = 0;
$OSMOD_WEAPONSTR   = 1;
$OSMOD_WEAPONACC   = 2;
$OSMOD_TRAINPROD   = 3;
$OSMOD_AXEUPGRADE  = 4;
$OSMOD_PICKUPGRADE = 5;
$OSMOD_MAXCOUNT    = 6;

// Game object global modifier values. Do NOT touch these.
$OSMOD_VALUES[$OSMOD_HITPOINTS]   = 0;
$OSMOD_VALUES[$OSMOD_WEAPONSTR]   = 0;
$OSMOD_VALUES[$OSMOD_WEAPONACC]   = 0;
$OSMOD_VALUES[$OSMOD_TRAINPROD]   = 1;
$OSMOD_VALUES[$OSMOD_AXEUPGRADE]  = 0;
$OSMOD_VALUES[$OSMOD_PICKUPGRADE] = 0;

////////////////////////////////////////////////////////////////////////////////
//! Returns the number of object modifiers
////////////////////////////////////////////////////////////////////////////////
function osGetModifierCount()
{
   return $OSMOD_MAXCOUNT;
}

////////////////////////////////////////////////////////////////////////////////
//! Returns the value of the modifier
//! \param %mod Modifier for which to retrieve value ($OSMOD_HITPOINTS, etc.)
////////////////////////////////////////////////////////////////////////////////
function osGetModifierValue(%mod)
{
   return $OSMOD_VALUES[%mod];
}

////////////////////////////////////////////////////////////////////////////////
//! Sets the value of the modifier
//! \param %mod Modifier for which to set value ($OSMOD_HITPOINTS, etc.)
//! \param %val Value for modifier
////////////////////////////////////////////////////////////////////////////////
function osSetModifierValue(%mod, %val)
{
   $OSMOD_VALUES[%mod] = %val;
   
   // Update existing objects
   if(%mod == $OSMOD_HITPOINTS) {
      osModifyAllBldgsHitPoints(%val);
   }
   else if(%mod == $OSMOD_WEAPONSTR || %mod == $OSMOD_WEAPONACC) {
      osModifyAllCombatUnits(%mod);
   }
   else if(%mod == $OSMOD_TRAINPROD) {
      osModifyAllTrainStations();
   }
   else if(%mod == $OSMOD_AXEUPGRADE) {
      osModifyAllWoodHuts();
   }
   else if(%mod == $OSMOD_PICKUPGRADE) {
      osModifyAllGoldMines();
   }
}

////////////////////////////////////////////////////////////////////////////////
//! Resets all modifiers to default values
////////////////////////////////////////////////////////////////////////////////
function osResetModifierValues()
{
   for(%i = 0; %i < $OSMOD_MAXCOUNT; %i++) {
      $OSMOD_VALUES[%i] = 0;
   }
}

////////////////////////////////////////////////////////////////////////////////
//! Modifies game object with all current modifiers
//! \param %obj Game object to modify
////////////////////////////////////////////////////////////////////////////////
function osModifySLGameObject(%obj)
{
   // Perform various modifications
   osModifyHitPoints(%obj);
   osModifyWeaponStrength(%obj);
   osModifyWeaponAcurracy(%obj);
   osModifyTrainStation(%obj);
   osModifyWoodHut(%obj);
   osModifyGoldMine(%obj);
}

////////////////////////////////////////////////////////////////////////////////
//! Modifies game object's hit points
//! \param %obj Game object to modify
////////////////////////////////////////////////////////////////////////////////
function osModifyHitPoints(%obj)
{
   // Needs to be a building
   if(!slgIsBuilding(%obj)) {
      return;
   }
   
   // Needs to be on team
   if(!%obj.isOnTeam($OST_PLAYER)) {
      return;
   }
   
   // Get hit point modifier
   %hpMod = osGetModifierValue($OSMOD_HITPOINTS);
   
   // Not modifying anything
   if(%hpMod == 0) {
      return;
   }
   
   // Update object's hitpoints
   %obj.healthMaxMod = %hpMod;
}

////////////////////////////////////////////////////////////////////////////////
//! Modifies the weapon strength for an object
//! \param %obj Object to modify
////////////////////////////////////////////////////////////////////////////////
function osModifyWeaponStrength(%obj)
{
   // Can't do this without purchase
   if(!csIsCmdUpgradePurchased($CSUP_GUN)) {
      return;
   }
   
   // Needs to be a character
   if(!slgIsCharacter(%obj)) {
      return;
   }
   
   // Needs to be on team
   if(!%obj.isOnTeam($OST_PLAYER)) {
      return;
   }
   
   // Needs to have a combat component
   %cmpCmbt = slgQueryInterface(%obj, $CID_COMBAT);
   if(!isObject(%cmpCmbt)) {
      return;
   }
   
   // Modify weapon strength
   %cmpCmbt.setWeaponStrengthMod(osGetModifierValue($OSMOD_WEAPONSTR));
}

////////////////////////////////////////////////////////////////////////////////
//! Modifies the weapon accuracy for an object
//! \param %obj Object to modify
////////////////////////////////////////////////////////////////////////////////
function osModifyWeaponAcurracy(%obj)
{
   // Can't do this without purchase
   if(!csIsCmdUpgradePurchased($CSUP_GUN)) {
      return;
   }
   
   // Needs to be a character
   if(!slgIsCharacter(%obj)) {
      return;
   }
   
   // Needs to be on team
   if(!%obj.isOnTeam($OST_PLAYER)) {
      return;
   }
   
   // Needs to have a combat component
   %cmpCmbt = slgQueryInterface(%obj, $CID_COMBAT);
   if(!isObject(%cmpCmbt)) {
      return;
   }
   
   // Modify weapon strength
   %cmpCmbt.setWeaponAccuracyMod(osGetModifierValue($OSMOD_WEAPONACC));
}

////////////////////////////////////////////////////////////////////////////////
//! Modifies the train station productivity
//! \param %obj Object to modify
////////////////////////////////////////////////////////////////////////////////
function osModifyTrainStation(%obj)
{
   // Upgrade should be purchased
   if(!csIsCmdUpgradePurchased($CSUP_TRACK)) {
      return;
   }
   
   // Object should be on player team
   if(!%obj.isOnTeam($OST_PLAYER)) {
      return;
   }
   
   // Object should be a train station
   if(!%obj.isOfType("trainstation")) {
      return;
   }
   
   // Needs to have a building component
   %cmpBldg = slgQueryInterface(%obj, $CID_BUILDING);
   if(!isObject(%cmpBldg)) {
      return;
   }
   
   // Increase productivity
   %cmpBldg.setProductionRate(osGetModifierValue($OSMOD_TRAINPROD));
}

////////////////////////////////////////////////////////////////////////////////
//! Modifies the gold mine productivity
//! \param %obj Object to modify
////////////////////////////////////////////////////////////////////////////////
function osModifyGoldMine(%obj) 
{
   // Upgrade needs to be purchased
   if(!csIsCmdUpgradePurchased($CSUP_PICK)) {
      return;
   }
   
   // Object should be on player team
   if(!%obj.isOnTeam($OST_PLAYER)) {
      return;
   }
   
   // Object should be a gold mine
   if(!%obj.isOfType("Goldmine")) {
      return;
   }
   
   // Needs to have a building component
   %cmpBldg = slgQueryInterface(%obj, $CID_BUILDING);
   if(!isObject(%cmpBldg)) {
      return;
   }
   
   // Increase production rate
   %cmpBldg.setProductionRate(osGetModifierValue($OSMOD_PICKUPGRADE));
}

////////////////////////////////////////////////////////////////////////////////
//! Modifies the wood hut productivity
//! \param %obj Object to modify
////////////////////////////////////////////////////////////////////////////////
function osModifyWoodHut(%obj)
{
   // Upgrade needs to be purchased
   if(!csIsCmdUpgradePurchased($CSUP_AXE)) {
      return;
   }
   
   // Object should be on player team
   if(!%obj.isOnTeam($OST_PLAYER)) {
      return;
   }
   
   // Object should be a wood hut
   if(!%obj.isOfType("Woodhut")) {
      return;
   }
   
   // Needs to have a building component
   %cmpBldg = slgQueryInterface(%obj, $CID_BUILDING);
   if(!isObject(%cmpBldg)) {
      return;
   }
   
   // Increase production rate
   %cmpBldg.setProductionRate(osGetModifierValue($OSMOD_AXEUPGRADE));
}

////////////////////////////////////////////////////////////////////////////////
//! Modifies all existing buildings hit points
//! \param %val Value to modify by
////////////////////////////////////////////////////////////////////////////////
function osModifyAllBldgsHitPoints(%val)
{
   // Get building list
   %bldgs = slgGetServerBuildingList();
   if(!isObject(%bldgs)) {
      return;
   }
 
   // Set hit point modifier on team buildings  
   for(%i = 0; %i < %bldgs.getSize(); %i++) {
      %bldg = %bldgs.getID(%i);
      if(!isObject(%bldg)) {
         continue;
      }
      if(!%bldg.isOnTeam($OST_PLAYER)) {
         continue;
      }
      
      if(%bldg.isLinkObj() == true) {
         continue;
      }
      
      %bldg.healthMaxMod = %val;
      %bldg.health += %bldg.healthMaxMod;
   }
}

////////////////////////////////////////////////////////////////////////////////
//! Modifies all existing combat units with the desired modifier
//! \param %mod Modifier ($OSMOD_WEAPONSTR or $OSMOD_WEAPONACC)
////////////////////////////////////////////////////////////////////////////////
function osModifyAllCombatUnits(%mod)
{
   // Get character list
   %chars = slgGetServerCharacterList();
   if(!isObject(%chars)) {
      return;
   }
   
   // Set weapon modifications on team characters
   for(%i = 0; %i < %chars.getSize(); %i++) {      
      // Update weapon strength modification
      if(%mod == $OSMOD_WEAPONSTR) {
         osModifyWeaponStrength(%chars.getID(%i));
      }
      // Update weapon accuracy modification
      else if(%mod == $OSMOD_WEAPONACC) {
         osModifyWeaponAcurracy(%chars.getID($i));
      }
   }
}

////////////////////////////////////////////////////////////////////////////////
//! Modifies all train stations in existence to produce more resources per tick
////////////////////////////////////////////////////////////////////////////////
function osModifyAllTrainStations()
{
   // Get buildings
   %bldgList = slgGetServerBuildingList();
   if(!isObject(%bldgList)) {
      return;
   }
   
   // For each building do modify it
   %bldgCount = %bldgList.getSize();
   for(%i = 0; %i < %bldgCount; %i++) {
      osModifyTrainStation(%bldgList.getID(%i));
   }
}

////////////////////////////////////////////////////////////////////////////////
//! Modifies all gold mines to produce more resources per tick
////////////////////////////////////////////////////////////////////////////////
function osModifyAllGoldMines()
{
   // Get buildings
   %bldgList = slgGetServerBuildingList();
   if(!isObject(%bldgList)) {
      return;
   }
   
   // For each building on our team do
   %bldgCount = %bldgList.getSize();
   for(%i = 0; %i < %bldgCount; %i++) {
      osModifyGoldMine(%bldgList.getID(%i));      
   }
}

////////////////////////////////////////////////////////////////////////////////
//! Modifies all wood huts to produce more resources per tick
////////////////////////////////////////////////////////////////////////////////
function osModifyAllWoodHuts()
{
   // Get buildings
   %bldgList = slgGetServerBuildingList();
   if(!isObject(%bldgList)) {
      return;
   }
   
   // For each building on our team do
   %bldgCount = %bldgList.getSize();
   for(%i = 0; %i < %bldgCount; %i++) {
      osModifyWoodHut(%bldgList.getID(%i));      
   }
}

// End svr_slgameobjmods.cs